//
//  RVProtocolParser.h
//  CommunicationModule
//
//  Created by Greg Harris on 7/24/12.
//  Copyright (c) 2012 Renewed Vision, LLC. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import "RVApplicationCommands.h"
#import "RVProtocolCommands.h"

extern NSString *RVProtocolParserSerializationKeyBundleName;
extern NSString *RVProtocolParserSerializationKeyBundleIdentifier;
extern NSString *RVProtocolParserSerializationKeyBehavior;

typedef enum _RVProtocolBehaviorMask {
	RVProtocolBehaviorNone = 0x00,
	RVProtocolBehaviorController = 0x01,
	RVProtocolBehaviorDevice = 0x02
	// this is a multi-value (power of 2) enum so the values will be 0x1,0x2,0x4,0x8,0x10,0x20,0x40,etc
} RVProtocolBehaviorMask;

@class RVHardwareCommunicator;
@protocol RVProtocolParserDelegate;

@protocol RVCommunicationPluginPrincipalClass <NSObject>
@required
+ (NSArray *) additionalClasses;
@end

@interface RVProtocolParser : NSObject <RVCommunicationPluginPrincipalClass> {
	RVHardwareCommunicator *hardwareCommunicator;
	id <RVProtocolParserDelegate> delegate;
	RVProtocolBehaviorMask behavior;
}

@property (retain) RVHardwareCommunicator *hardwareCommunicator;
@property (assign) id <RVProtocolParserDelegate> delegate;
@property (assign) RVProtocolBehaviorMask behavior;

/*
 * Returns the name of the protocol to be used for display purposes
 */
+ (NSString *) supportedProtocol;

/*
 * Returns an icon for the protocol to be used for display purposes
 */
+ (NSImage *) supportedProtocolIcon;

/*
 * Returns a description of the protocol to be used for display purposes
 */
+ (NSString *) supportedProtocolDescription;

/*
 * Returns an array of Class objects corresponding to the RVHardwareCommunicator subclasses that the protocol supports.
 */
+ (NSArray *) supportedHardwareCommunicators;

/*
 * Returns a multi-value enumeration (mask) describing the possible protocol behaviors.
 */
+ (NSUInteger) protocolBehaviorMask;

/*
 * Returns a dictionary of dictionaries with the specified in RVProtocolCommands.h. 
 * These dictionaries define commands that can be sent to the device. 
 * The keys of the top level dictionary are the names of the commands that each value (also a dictionary) contains.
 */
- (NSDictionary *) protocolCommands;

/*
 * Triggers a command defined by a dictionary with the keys described in RVProcotolCommands.h. 
 * These are nearly the same keys you get from a command dictionary in -[RVProtocolParser protocolCommands]. 
 * Generally, you'll modify that dictionary to add the parameter values from the UI and return that dictionary to this method.
 */
- (void) sendProtocolCommand:(NSDictionary *)cmdDictionary;


/*
 * Returns a view that allows the parser to display a control interface that directly interacts with the hardware device.
 */
- (NSView *) controlView;

/*
 * Returns an XML Element that serializes the current hardware and protocol settings
 */
- (NSXMLElement *)serializedParser;

/*
 * Returns the acceptable protocol behavior mask for a given hardware communicator class. By default, it returns [[self class] protocolBehaviorMask] .
 */
+ (NSUInteger) protocolBehaviorMaskForHardware:(Class)hwClass;

@end


@protocol RVProtocolParserDelegate <NSObject>

/*
 * Your parser object will call this method on the parser's delegate, which should be the CommunicationModuleController, which then calls the command on your application's delegate
 */
- (id) executeApplicationCommand:(NSString *)cmd withParams:(NSDictionary *)params;

@end


